import * as state from './state.js';
import * as ui from './ui.js';
import * as ipc from './ipc.js';
import { playSound, navigateSound, confirmSound, cancelSound, backgroundMusic, attractMusicPlayer, t, applyTranslationsToDOM, anyModalOpen, fadeOutAndStop, shuffleArray } from './utils.js';

function startAttractMode() {
    if (state.appSettings.attractModeEnabled !== 'true' || anyModalOpen() || state.isProcessActive) {
        return;
    }

    ui.attractModeOverlay.classList.add('visible');
    fadeOutAndStop(backgroundMusic);

    const playlist = shuffleArray(state.availableBgmTracks.filter(track => track.path !== null && track.name !== 'Music Off' && track.name !== 'Default'));
    state.setAttractPlaylist(playlist);
    state.setAttractPlaylistIndex(0);

    const playNextAttractSong = () => {
        if (state.attractPlaylist.length === 0) return;
        
        const track = state.attractPlaylist[state.attractPlaylistIndex];
        attractMusicPlayer.src = `file:///${track.path.replace(/\\/g, '/')}`;
        attractMusicPlayer.volume = state.appSettings.bgmVolumeLevel / 10 * 0.5;
        attractMusicPlayer.play().catch(e => console.error("Attract BGM failed to play:", e));

        state.setAttractPlaylistIndex((state.attractPlaylistIndex + 1) % state.attractPlaylist.length);
    };

    attractMusicPlayer.onended = playNextAttractSong;
    playNextAttractSong();

    const interval = setInterval(() => {
        if (anyModalOpen() || state.isProcessActive) {
            resetInactivityTimer();
            return;
        }
        ui.updateSelection(state.selectedIndex + 1, false);
    }, state.ATTRACT_MODE_SCROLL_INTERVAL);
    state.setAttractModeInterval(interval);
}

export function resetInactivityTimer() {
    ui.attractModeOverlay.classList.remove('visible');
    clearTimeout(state.inactivityTimer);
    clearInterval(state.attractModeInterval);
    state.setAttractModeInterval(null);

    if (!attractMusicPlayer.paused) {
        fadeOutAndStop(attractMusicPlayer);
        if (state.appSettings.bgmPath && state.appSettings.bgmPath !== "off") {
            backgroundMusic.volume = state.appSettings.bgmVolumeLevel / 10 * 0.5;
            backgroundMusic.currentTime = 0;
            backgroundMusic.play().catch(e => {});
        }
    }

    if (state.appSettings.attractModeEnabled === 'true' && !state.isProcessActive) {
        const timer = setTimeout(startAttractMode, state.ATTRACT_MODE_TIMEOUT);
        state.setInactivityTimer(timer);
    }
}

function handleUserInteraction() {
    if (!state.hasUserInteracted) {
        state.setHasUserInteracted(true);
        ui.showNotification();
    }
    resetInactivityTimer();
}

function dispatchKeyEvent(keyName) {
    window.dispatchEvent(new KeyboardEvent('keydown', { key: keyName }));
}

function checkButton(gamepad, buttonIndex, stateKey, action) {
    if (gamepad.buttons[buttonIndex]?.pressed && !state.previousButtonStates[stateKey]) {
        handleUserInteraction();
        action();
    }
    state.previousButtonStates[stateKey] = gamepad.buttons[buttonIndex]?.pressed;
}

function processDirection(buttonOrAxisActive, stateKey, eventKey) {
    const now = Date.now();
    if (buttonOrAxisActive) {
        if (!state.previousButtonStates[stateKey]) {
            handleUserInteraction();
            dispatchKeyEvent(eventKey);
            state.previousButtonStates[stateKey] = true;
            state.buttonPressedTime[stateKey] = now;
            state.lastDirectionalInputTime[stateKey] = now;
        } else if (now - state.buttonPressedTime[stateKey] >= state.INITIAL_REPEAT_DELAY) {
            if (now - state.lastDirectionalInputTime[stateKey] >= state.REPEAT_DELAY) {
                dispatchKeyEvent(eventKey);
                state.lastDirectionalInputTime[stateKey] = now;
            }
        }
    } else {
        state.previousButtonStates[stateKey] = false;
        state.buttonPressedTime[stateKey] = 0;
    }
}

function focusNextTopMenu() {
    const openFuncs = [ui.openConfigMenu, ui.openAudioMenu, ui.openDisplayMenu, ui.openNetworkMenu, ui.openPowerMenu];
    const closeFuncs = [ui.closeConfigMenu, ui.closeAudioMenu, ui.closeDisplayMenu, ui.closeNetworkMenu, ui.closePowerMenu];
    
    closeFuncs[state.topMenuSelectionIndex](false);
    state.setTopMenuSelectionIndex((state.topMenuSelectionIndex + 1) % ui.topMenuItems.length);
    ui.updateTopMenuSelection();
    playSound(navigateSound);
    openFuncs[state.topMenuSelectionIndex](false);
}

function focusPreviousTopMenu() {
    const openFuncs = [ui.openConfigMenu, ui.openAudioMenu, ui.openDisplayMenu, ui.openNetworkMenu, ui.openPowerMenu];
    const closeFuncs = [ui.closeConfigMenu, ui.closeAudioMenu, ui.closeDisplayMenu, ui.closeNetworkMenu, ui.closePowerMenu];

    closeFuncs[state.topMenuSelectionIndex](false);
    state.setTopMenuSelectionIndex((state.topMenuSelectionIndex - 1 + ui.topMenuItems.length) % ui.topMenuItems.length);
    ui.updateTopMenuSelection();
    playSound(navigateSound);
    openFuncs[state.topMenuSelectionIndex](false);
}

function pollGamepads() {
    if (!document.hasFocus()) { requestAnimationFrame(pollGamepads); return; }
    const gamepad = navigator.getGamepads()[0]; if (!gamepad) { requestAnimationFrame(pollGamepads); return; } const A_BUTTON = 0, B_BUTTON = 1, SELECT_BUTTON = 8, START_BUTTON = 9, DPAD_UP = 12, DPAD_DOWN = 13, DPAD_LEFT = 14, DPAD_RIGHT = 15, LB_BUTTON = 4, RB_BUTTON = 5; const up = gamepad.buttons[DPAD_UP]?.pressed; const down = gamepad.buttons[DPAD_DOWN]?.pressed; const left = gamepad.buttons[DPAD_LEFT]?.pressed; const right = gamepad.buttons[DPAD_RIGHT]?.pressed;
    if (state.isProcessActive) { checkButton(gamepad, B_BUTTON, 'B', () => ipc.cancelEmuProcess()); }
    else if (state.isUsernameModalOpen || state.isRaLoginModalOpen) { processDirection(up, 'Up', 'ArrowUp'); processDirection(down, 'Down', 'ArrowDown'); processDirection(left, 'Left', 'ArrowLeft'); processDirection(right, 'Right', 'ArrowRight'); checkButton(gamepad, A_BUTTON, 'A', () => dispatchKeyEvent('Enter')); checkButton(gamepad, B_BUTTON, 'B', () => dispatchKeyEvent('Escape')); }
    else if (state.isMenuOpen || state.isSaveStateModalOpen) { processDirection(up, 'Up', 'ArrowUp'); processDirection(down, 'Down', 'ArrowDown'); checkButton(gamepad, A_BUTTON, 'A', () => dispatchKeyEvent('Enter')); checkButton(gamepad, B_BUTTON, 'B', () => dispatchKeyEvent('Escape')); }
    else if (state.isDisplayMenuOpen || state.isConfigMenuOpen || state.isNetworkMenuOpen || state.isAudioMenuOpen || state.isPowerMenuOpen) { processDirection(up, 'Up', 'ArrowUp'); processDirection(down, 'Down', 'ArrowDown'); processDirection(left, 'Left', 'ArrowLeft'); processDirection(right, 'Right', 'ArrowRight'); checkButton(gamepad, LB_BUTTON, 'LB', focusPreviousTopMenu); checkButton(gamepad, RB_BUTTON, 'RB', focusNextTopMenu); checkButton(gamepad, A_BUTTON, 'A', () => dispatchKeyEvent('Enter')); checkButton(gamepad, B_BUTTON, 'B', () => dispatchKeyEvent('Escape')); }
    else if (state.isLobbyOpen) { processDirection(up, 'Up', 'ArrowUp'); processDirection(down, 'Down', 'ArrowDown'); checkButton(gamepad, A_BUTTON, 'A', () => dispatchKeyEvent('Enter')); checkButton(gamepad, B_BUTTON, 'B', () => dispatchKeyEvent('Escape')); }
    else if (state.isSaveStateMenuOpen) { processDirection(up, 'Up', 'ArrowUp'); processDirection(left, 'Left', 'ArrowLeft'); processDirection(right, 'Right', 'ArrowRight'); checkButton(gamepad, A_BUTTON, 'A', () => dispatchKeyEvent('Enter')); checkButton(gamepad, B_BUTTON, 'B', () => dispatchKeyEvent('Escape')); }
    else if (state.isTopMenuFocused) { processDirection(down, 'Down', 'ArrowDown'); processDirection(left, 'Left', 'ArrowLeft'); processDirection(right, 'Right', 'ArrowRight'); checkButton(gamepad, LB_BUTTON, 'LB', () => dispatchKeyEvent('ArrowLeft')); checkButton(gamepad, RB_BUTTON, 'RB', () => dispatchKeyEvent('ArrowRight')); checkButton(gamepad, A_BUTTON, 'A', () => dispatchKeyEvent('Enter')); checkButton(gamepad, B_BUTTON, 'B', () => dispatchKeyEvent('Escape')); }
    else if (state.isAlphaBarOpen) { processDirection(up, 'Up', 'ArrowUp'); processDirection(down, 'Down', 'ArrowDown'); processDirection(left, 'Left', 'ArrowLeft'); processDirection(right, 'Right', 'ArrowRight'); checkButton(gamepad, A_BUTTON, 'A', () => dispatchKeyEvent('Enter')); checkButton(gamepad, B_BUTTON, 'B', () => dispatchKeyEvent('Escape')); checkButton(gamepad, START_BUTTON, 'Start', () => dispatchKeyEvent('s')); }
    else { processDirection(up, 'Up', 'ArrowUp'); processDirection(down, 'Down', 'ArrowDown'); processDirection(left, 'Left', 'ArrowLeft'); processDirection(right, 'Right', 'ArrowRight'); checkButton(gamepad, LB_BUTTON, 'LB', ui.focusTopMenu); checkButton(gamepad, RB_BUTTON, 'RB', ui.focusTopMenu); checkButton(gamepad, A_BUTTON, 'A', () => dispatchKeyEvent('Enter')); checkButton(gamepad, START_BUTTON, 'Start', () => dispatchKeyEvent('s')); checkButton(gamepad, SELECT_BUTTON, 'Select', () => dispatchKeyEvent('Backspace')); } requestAnimationFrame(pollGamepads);
}

function handleMenuInput(key) { if (key === 'ArrowUp' || key === 'ArrowDown') { playSound(navigateSound); if (key === 'ArrowUp') state.setMenuSelectionIndex((state.menuSelectionIndex - 1 + ui.launchMenuButtons.length) % ui.launchMenuButtons.length); if (key === 'ArrowDown') state.setMenuSelectionIndex((state.menuSelectionIndex + 1) % ui.launchMenuButtons.length); ui.updateMenuSelection(); } else if (key === 'Enter') { playSound(confirmSound); ui.launchMenuButtons[state.menuSelectionIndex].click(); } else if (key === 'Escape' || key === 'Backspace') { ui.closeLaunchMenu(); } }
function handleCarouselInput(key) { if (backgroundMusic.paused && ['ArrowRight', 'ArrowLeft', 'ArrowDown', 'ArrowUp', 'Enter'].includes(key) && state.appSettings.bgmPath) { backgroundMusic.currentTime = 0; backgroundMusic.play().catch(e => {}); } if (key === 'ArrowRight') { ui.updateSelection(state.selectedIndex + 1); } else if (key === 'ArrowLeft') { ui.updateSelection(state.selectedIndex - 1); } else if (key === 'ArrowDown') { if (state.appSettings.savestate_auto_index === 'true') ui.openSaveStateMenu(); } else if (key === 'ArrowUp') { ui.focusTopMenu(); } else if (key === 'Enter') { ui.openLaunchMenu(); } else if (key === 's') { ui.openAlphaBar(); } else if (key === 'Backspace') { ui.openLobbyMenu(); } }
function handleAlphaBarInput(key) {
    const jumpToSelectedLetter = () => {
        const letter = state.ALPHABET[state.alphaSelectionIndex]; const isNumeric = letter === '#';
        const foundIndex = state.games.findIndex(game => { const firstChar = game.baseName.charAt(0); if (isNumeric) return !isNaN(parseInt(firstChar)); return firstChar.toUpperCase() === letter; });
        if (foundIndex !== -1) ui.updateSelection(foundIndex, true);
    };
    if (key === 'ArrowRight') { state.setAlphaSelectionIndex((state.alphaSelectionIndex + 1) % state.ALPHABET.length); playSound(navigateSound); ui.updateAlphaSelection(); jumpToSelectedLetter(); }
    else if (key === 'ArrowLeft') { state.setAlphaSelectionIndex((state.alphaSelectionIndex - 1 + state.ALPHABET.length) % state.ALPHABET.length); playSound(navigateSound); ui.updateAlphaSelection(); jumpToSelectedLetter(); }
    else if (key === 'Enter') { ui.closeAlphaBar(false); } 
    else if (key === 'Escape' || key === 'Backspace' || key === 'ArrowUp' || key === 's') { ui.closeAlphaBar(); }
}
function handleSaveStateMenuInput(key) { if (key === 'ArrowUp' || key === 'Escape' || key === 'Backspace') { ui.closeSaveStateMenu(); } else if (key === 'ArrowLeft') { const newIndex = Math.max(0, state.saveStateSelectionIndex - 1); if (newIndex !== state.saveStateSelectionIndex) { state.setSaveStateSelectionIndex(newIndex); ui.updateSaveStateSelection(); } } else if (key === 'ArrowRight') { const newIndex = Math.min(3, state.saveStateSelectionIndex + 1); if (newIndex !== state.saveStateSelectionIndex) { state.setSaveStateSelectionIndex(newIndex); ui.updateSaveStateSelection(); } } else if (key === 'Enter') { if (state.currentSaveStates[state.saveStateSelectionIndex] !== null) { ui.openSaveStateModal(); } } }
function handleSaveStateModalInput(key) { if (key === 'ArrowUp' || key === 'ArrowDown') { playSound(navigateSound); if (key === 'ArrowUp') state.setSaveStateModalSelectionIndex((state.saveStateModalSelectionIndex - 1 + ui.saveStateModalButtons.length) % ui.saveStateModalButtons.length); else state.setSaveStateModalSelectionIndex((state.saveStateModalSelectionIndex + 1) % ui.saveStateModalButtons.length); ui.updateSaveStateModalSelection(); } else if (key === 'Enter') { playSound(confirmSound); ui.saveStateModalButtons[state.saveStateModalSelectionIndex].click(); } else if (key === 'Escape' || key === 'Backspace') { ui.closeSaveStateModal(); } }
function handleTopMenuInput(key) { if (key === 'ArrowRight') { state.setTopMenuSelectionIndex((state.topMenuSelectionIndex + 1) % ui.topMenuItems.length); playSound(navigateSound); ui.updateTopMenuSelection(); } else if (key === 'ArrowLeft') { state.setTopMenuSelectionIndex((state.topMenuSelectionIndex - 1 + ui.topMenuItems.length) % ui.topMenuItems.length); playSound(navigateSound); ui.updateTopMenuSelection(); } else if (key === 'ArrowDown' || key === 'Escape' || key === 'Backspace') { ui.unfocusTopMenu(); } else if (key === 'Enter') { if (state.topMenuSelectionIndex === 0) { ui.openConfigMenu(); } else if (state.topMenuSelectionIndex === 1) { ui.openAudioMenu(); } else if (state.topMenuSelectionIndex === 2) { ui.openDisplayMenu(); } else if (state.topMenuSelectionIndex === 3) { ui.openNetworkMenu(); } else if (state.topMenuSelectionIndex === 4) { ui.openPowerMenu(); } } }
function handleLobbyInput(key) { if (state.lobbyGames.length === 0) { if (key === 'Enter' || key === 'Escape' || key === 'Backspace') { ui.closeLobbyMenu(); } return; } if (key === 'ArrowUp') { state.setLobbySelectionIndex((state.lobbySelectionIndex - 1 + state.lobbyGames.length) % state.lobbyGames.length); playSound(navigateSound); ui.updateLobbySelection(); } else if (key === 'ArrowDown') { state.setLobbySelectionIndex((state.lobbySelectionIndex + 1) % state.lobbyGames.length); playSound(navigateSound); ui.updateLobbySelection(); } else if (key === 'Enter') { const selectedLobbyGame = state.lobbyGames[state.lobbySelectionIndex]; const gameName = selectedLobbyGame.game_name; const gameFilename = `${gameName}.zip`; const gamePath = `${state.APP_PATHS.gamesDir}/${gameFilename}`.replace(/\\/g, '/'); playSound(confirmSound); ui.closeLobbyMenu(); ipc.joinNetplayGame({ gamePath: gamePath, ip: selectedLobbyGame.ip, port: selectedLobbyGame.port, mitm_ip: selectedLobbyGame.mitm_ip, mitm_session: selectedLobbyGame.mitm_session }); ui.showProgressOverlay(t('progress_joining_game', { gameName })); } else if (key === 'Escape' || key === 'Backspace') { ui.closeLobbyMenu(); } }
function handlePowerMenuInput(key) { if (key === 'ArrowUp' || key === 'ArrowDown') { playSound(navigateSound); if (key === 'ArrowUp') { state.setPowerMenuSelectionIndex((state.powerMenuSelectionIndex - 1 + ui.powerMenuButtons.length) % ui.powerMenuButtons.length); } else if (key === 'ArrowDown') { state.setPowerMenuSelectionIndex((state.powerMenuSelectionIndex + 1) % ui.powerMenuButtons.length); } ui.updatePowerMenuSelection(); } else if (key === 'Enter') { playSound(confirmSound); ui.powerMenuButtons[state.powerMenuSelectionIndex].click(); } else if (key === 'Escape' || key === 'Backspace') { ui.closePowerMenu(); } }
function handleAudioMenuInput(key) { if (key === 'Escape' || key === 'Backspace') { ui.closeAudioMenu(); return; } if (key === 'ArrowUp' || key === 'ArrowDown') { state.setAudioMenuSelectionIndex((state.audioMenuSelectionIndex + 1) % ui.audioMenuOptions.length); playSound(navigateSound); ui.updateAudioMenuSelection(); return; } if (key !== 'ArrowLeft' && key !== 'ArrowRight') { return; } if (state.audioMenuSelectionIndex === 0) { if (key === 'ArrowLeft') { state.appSettings.bgmVolumeLevel = Math.max(0, state.appSettings.bgmVolumeLevel - 1); } else { state.appSettings.bgmVolumeLevel = Math.min(10, state.appSettings.bgmVolumeLevel + 1); } backgroundMusic.volume = state.appSettings.bgmVolumeLevel / 10 * 0.5; ui.renderBgmVolumeBar(); playSound(navigateSound); } else if (state.audioMenuSelectionIndex === 1) { if (key === 'ArrowLeft') { ui.changeBgm(-1); } else { ui.changeBgm(1); } } }
async function handleDisplayMenuInput(key) {
    if (key === 'Escape' || key === 'Backspace') { ui.closeDisplayMenu(); return; }
    if (key === 'ArrowUp' || key === 'ArrowDown') {
        state.setDisplayMenuSelectionIndex((state.displayMenuSelectionIndex + (key === 'ArrowDown' ? 1 : -1) + ui.displayMenuOptions.length) % ui.displayMenuOptions.length);
        playSound(navigateSound); ui.updateDisplayMenuSelection(); return;
    }
    if (key !== 'ArrowLeft' && key !== 'ArrowRight') return;
    playSound(navigateSound);
    if (state.displayMenuSelectionIndex === 0) {
        state.appSettings.startFullscreen = !state.appSettings.startFullscreen;
        ipc.setFullscreen(state.appSettings.startFullscreen);
        ui.populateDisplayMenu();
    }
    else if (state.displayMenuSelectionIndex === 1) { const currentIndex = state.availableLanguages.findIndex(l => l.code === state.appSettings.language); const direction = key === 'ArrowRight' ? 1 : -1; const nextIndex = (currentIndex + direction + state.availableLanguages.length) % state.availableLanguages.length; const newLangCode = state.availableLanguages[nextIndex].code; state.appSettings.language = newLangCode; state.appSettings.user_language = state.RETROARCH_LANG_MAP[newLangCode] || '0'; await ipc.loadTranslations(state.appSettings.language); applyTranslationsToDOM(); ui.populateDisplayMenu(); }
    else if (state.displayMenuSelectionIndex === 2) { state.appSettings.attractModeEnabled = state.appSettings.attractModeEnabled === 'true' ? 'false' : 'true'; ui.attractModeDisplay.textContent = state.appSettings.attractModeEnabled === 'true' ? t('common_on') : t('common_off'); resetInactivityTimer(); }
}
function handleNetworkMenuInput(key) { if (key === 'Escape' || key === 'Backspace') { ui.closeNetworkMenu(); return; } if (key === 'ArrowUp' || key === 'ArrowDown') { state.setNetworkMenuSelectionIndex((state.networkMenuSelectionIndex + (key === 'ArrowDown' ? 1 : -1) + ui.networkMenuOptions.length) % ui.networkMenuOptions.length); playSound(navigateSound); ui.updateNetworkMenuSelection(); return; } const selectedOption = ui.networkMenuOptions[state.networkMenuSelectionIndex]; if (key === 'Enter') { if (!selectedOption.classList.contains('disabled')) { playSound(confirmSound); const target = selectedOption.querySelector('button') || selectedOption; target.click(); } return; } if (key !== 'ArrowLeft' && key !== 'ArrowRight') return; playSound(navigateSound); if (state.networkMenuSelectionIndex === 1) { const isRaEnabled = state.appSettings.cheevos_enable === 'true'; state.appSettings.cheevos_enable = isRaEnabled ? "false" : "true"; ui.populateNetworkMenu(); } else if (state.networkMenuSelectionIndex === 4) { const direction = key === 'ArrowRight' ? 1 : -1; const newIndex = (state.currentRelayIndex + direction + state.relayOptions.length) % state.relayOptions.length; state.setCurrentRelayIndex(newIndex); ui.updateRelayServerDisplay(); } }
function handleUsernameModalInput(key) { if (key === 'Escape') { ui.usernameModalInputs[2].click(); return; } if (key === 'Enter') { if (state.usernameModalSelectionIndex === 0) { ui.usernameModalInputs[0].focus(); } else { ui.usernameModalInputs[state.usernameModalSelectionIndex].click(); } } if (key === 'ArrowUp' || key === 'ArrowDown') { state.setUsernameModalSelectionIndex((state.usernameModalSelectionIndex + (key === 'ArrowDown' ? 1 : -1) + ui.usernameModalInputs.length) % ui.usernameModalSelectionIndex); playSound(navigateSound); ui.updateUsernameModalSelection(); } else if (key === 'ArrowLeft' || key === 'ArrowRight') { if (state.usernameModalSelectionIndex >= 1) { state.setUsernameModalSelectionIndex((state.usernameModalSelectionIndex % 2 === 1) ? 2 : 1); playSound(navigateSound); ui.updateUsernameModalSelection(); } } }
function handleRaLoginModalInput(key) { if (key === 'Escape') { ui.raLoginModalButtons[1].click(); return; } if (key === 'Enter') { if (state.raLoginModalSelectionIndex < 2) { ui.raLoginModalInputs[state.raLoginModalSelectionIndex].focus(); } else { ui.raLoginModalInputs[state.raLoginModalSelectionIndex].click(); } } if (key === 'ArrowUp' || key === 'ArrowDown') { state.setRaLoginModalSelectionIndex((state.raLoginModalSelectionIndex + (key === 'ArrowDown' ? 1 : -1) + ui.raLoginModalInputs.length) % ui.raLoginModalInputs.length); playSound(navigateSound); ui.updateRaLoginModalSelection(); } else if (key === 'ArrowLeft' || key === 'ArrowRight') { if (state.raLoginModalSelectionIndex >= 2) { state.setRaLoginModalSelectionIndex((state.raLoginModalSelectionIndex % 2 === 0) ? 3 : 2); playSound(navigateSound); ui.updateRaLoginModalSelection(); } } }
function handleConfigMenuInput(key) { if (key === 'Escape' || key === 'Backspace') { ui.closeConfigMenu(); return; } if (key === 'ArrowUp' || key === 'ArrowDown') { state.setConfigMenuSelectionIndex((state.configMenuSelectionIndex + (key === 'ArrowDown' ? 1 : -1) + ui.configMenuOptions.length) % ui.configMenuOptions.length); playSound(navigateSound); ui.updateConfigMenuSelection(); return; } if (state.configMenuSelectionIndex === 6) { if (key === 'Enter') { ui.configMenuOptions[6].click(); } return; } if (key !== 'ArrowLeft' && key !== 'ArrowRight') return; playSound(navigateSound); const direction = key === 'ArrowRight' ? 1 : -1; switch (state.configMenuSelectionIndex) { case 0: const isSaveEnabled = state.appSettings.savestate_auto_index === 'true'; state.appSettings.savestate_auto_index = isSaveEnabled ? "false" : "true"; state.appSettings.savestate_auto_load = isSaveEnabled ? "false" : "true"; state.appSettings.savestate_max_keep = isSaveEnabled ? "0" : "4"; break; case 1: const aspects = ['4:3', '16:9', 'Pixel Perfect']; let currentAspectIndex = aspects.indexOf(ui.aspectRatioDisplay.textContent); currentAspectIndex = (currentAspectIndex + direction + aspects.length) % aspects.length; state.appSettings.aspect_ratio_index = state.aspectValues[aspects[currentAspectIndex]]; break; case 2: state.appSettings.video_fullscreen = state.appSettings.video_fullscreen === 'true' ? 'false' : 'true'; break; case 3: state.appSettings.input_overlay_enable = state.appSettings.input_overlay_enable === 'true' ? 'false' : 'true'; break; case 4: const strengths = ['Low', 'Normal', 'High']; let currentStrengthIndex = strengths.findIndex(s => t(`configmenu_strength_${s.toLowerCase()}`) === ui.scanlinesStrengthDisplay.textContent); currentStrengthIndex = (currentStrengthIndex + direction + strengths.length) % strengths.length; state.appSettings.input_overlay_opacity = state.strengthValues[strengths[currentStrengthIndex]]; break; case 5: const isRwEnabled = state.appSettings.rewind_enable === 'true'; state.appSettings.rewind_enable = isRwEnabled ? 'false' : 'true'; state.appSettings.input_hold_fast_forward_axis = isRwEnabled ? 'nul' : '+5'; break; } ui.populateConfigMenu(); }

export function initializeInput() {
    resetInactivityTimer();
    window.addEventListener('keydown', (e) => {
        handleUserInteraction();
        if (state.isProcessActive) { if (e.key === 'Escape' || e.key === 'Backspace') { ipc.cancelEmuProcess(); } return; }
        if (state.isRaLoginModalOpen) { if(e.key !== 'Backspace') handleRaLoginModalInput(e.key); return; }
        if (state.isUsernameModalOpen) { if(e.key !== 'Backspace') handleUsernameModalInput(e.key); return; }
        if (state.isMenuOpen) { handleMenuInput(e.key); } 
        else if (state.isPowerMenuOpen) { handlePowerMenuInput(e.key); } else if (state.isAudioMenuOpen) { handleAudioMenuInput(e.key); }
        else if (state.isDisplayMenuOpen) { handleDisplayMenuInput(e.key); } else if (state.isNetworkMenuOpen) { handleNetworkMenuInput(e.key); }
        else if (state.isConfigMenuOpen) { handleConfigMenuInput(e.key); } else if (state.isLobbyOpen) { handleLobbyInput(e.key); }
        else if (state.isSaveStateModalOpen) { handleSaveStateModalInput(e.key); } else if (state.isSaveStateMenuOpen) { handleSaveStateMenuInput(e.key); }
        else if (state.isTopMenuFocused) { handleTopMenuInput(e.key); } else if (state.isAlphaBarOpen) { handleAlphaBarInput(e.key); }
        else { handleCarouselInput(e.key); }
    });
    
    const getSelectedGamePath = () => `${state.APP_PATHS.gamesDir}/${state.games[state.selectedIndex].zipName}`.replace(/\\/g, '/');

    document.getElementById('start-game-button').addEventListener('click', () => { ipc.launchGame({ gamePath: getSelectedGamePath() }); ui.showProgressOverlay(t('progress_initializing')); });
    document.getElementById('netplay-button').addEventListener('click', () => { ipc.netplayHost({ gamePath: getSelectedGamePath() }); ui.showProgressOverlay(t('progress_starting_netplay')); });
    document.getElementById('download-button').addEventListener('click', () => { ipc.saveGame(getSelectedGamePath()); ui.showProgressOverlay(t('progress_saving_collection')); });
    document.getElementById('exit-app-button').addEventListener('click', () => { ipc.exitApp(); });
    document.getElementById('suspend-pc-button').addEventListener('click', () => { backgroundMusic.pause(); backgroundMusic.currentTime = 0; ui.closePowerMenu(); ipc.suspendPc(); });
    document.getElementById('power-off-pc-button').addEventListener('click', () => { ui.closePowerMenu(); ipc.powerOffPc(); });
    document.getElementById('open-lobby-button').addEventListener('click', () => { ui.closeNetworkMenu(); ui.openLobbyMenu(); });
    document.getElementById('netplay-username-button').addEventListener('click', () => { ui.openUsernameModal(); });
    document.getElementById('download-collection-button').addEventListener('click', () => { ui.closeNetworkMenu(); ipc.downloadCollection(); ui.showProgressOverlay(t('progress_preparing_download')); });
    document.getElementById('username-save-button').addEventListener('click', () => { playSound(confirmSound); state.appSettings.netplayUsername = ui.usernameInput.value || "Player1"; ipc.saveAllSettings(state.appSettings); ui.closeUsernameModal(); });
    document.getElementById('username-cancel-button').addEventListener('click', () => { playSound(cancelSound); ui.closeUsernameModal(); });
    document.getElementById('ra-login-button').addEventListener('click', () => { ui.openRaLoginModal(); });
    document.getElementById('ra-login-save-button').addEventListener('click', () => { playSound(confirmSound); state.appSettings.cheevos_username = ui.raUsernameInput.value; state.appSettings.cheevos_password = ui.raPasswordInput.value; ipc.saveAllSettings(state.appSettings); ui.closeRaLoginModal(false); });
    document.getElementById('ra-login-cancel-button').addEventListener('click', () => { playSound(cancelSound); ui.closeRaLoginModal(true); });
    document.getElementById('restore-defaults-button').addEventListener('click', async () => { playSound(confirmSound); for (const key in state.IN_GAME_DEFAULTS) { state.appSettings[key] = state.IN_GAME_DEFAULTS[key]; } ui.populateConfigMenu(); });
    
    ui.saveStateModalButtons[0].addEventListener('click', () => { const stateData = state.currentSaveStates[state.saveStateSelectionIndex]; ipc.launchGame({ gamePath: getSelectedGamePath(), stateSlot: stateData ? stateData.fileSlot : null }); ui.showProgressOverlay(t('progress_loading_savestate')); });
    ui.saveStateModalButtons[1].addEventListener('click', async () => { const stateData = state.currentSaveStates[state.saveStateSelectionIndex]; if (!stateData) return; const gameBaseName = state.games[state.selectedIndex].zipName.replace(/\.zip$/i, ''); await ipc.manageSaveState({ action: 'delete', gameBaseName, slot: stateData.fileSlot }); ui.closeSaveStateModal(); await ui.populateSaveStateSlots(); delete state.saveStateCache[gameBaseName]; ui.updateSuspendPointsUI(state.selectedIndex); });
    ui.saveStateModalButtons[2].addEventListener('click', () => { ui.closeSaveStateModal(); });

    requestAnimationFrame(pollGamepads);
}